/*:
 * @target MZ
 * @plugindesc RSTH_IH: サバイバルゲームシステムプラグイン
 * @author © 2025 ReSera_りせら（@MOBIUS1001）
 *
 * このソースコードは無断での転載、複製、改変、再配布、商用利用を固く禁じます。
 * 禁止事項の例：
 * - 本ファイルの全部または一部を許可なくコピー、再配布すること
 * - 本ファイルを改変して配布すること
 * - 商用目的での利用
 */

(() => {

    "use strict";

    // ログ出力制御フラグ（trueでログ出力、falseで抑制）
    //const RSTH_DEBUG_LOG = true;
    const RSTH_DEBUG_LOG = false;


    window.RSTH_IH.Scene_UnshoShop = class extends Scene_MenuBase {
        constructor() {
            super();
            this.margin = 120;
        }

        initialize() {
            super.initialize();
        }

        create() {
            super.create();
            this.createTitle();
            this.createUnshoPWindow();
            this.createKyoukaUnshoPWindow();
            this.createUnshoItemsWindow();
            this.createConfirmWindow(); // ← 追加
            this.createSimpleMessageWindow();
            this.createHintText();

            this._unshoItemsWindow.activate();
        }

        createBackground() {
            this._backgroundSprite = new window.RSTH_IH.Sprite_CharacterSelectBackground();
            this.addChild(this._backgroundSprite);
        }

        createTitle() {
            this._title = new window.RSTH_IH.Sprite_VerticalTitleText("PowerUP");
            this.addChild(this._title);
        }


        createHintText() {
            const bitmap = new Bitmap(1200, 36);
            const x = 240;

            bitmap.fontSize = 24;
            bitmap.textColor = "#FFFFFF";
            bitmap.drawText("↑↓→←：項目選択", 0, 0, 300, 36, "left");
            bitmap.drawText("Enter：項目決定", x, 0, 300, 36, "left");
            bitmap.drawText("Esc：前の画面、項目へ", x * 2, 0, 300, 36, "left");
            bitmap.drawText("→ ←：ゲージ増減", x * 4, 0, 300, 36, "left");

            this._HintSprite = new Sprite(bitmap);
            this._HintSprite.x = 120;
            this._HintSprite.y = 862;
            this._HintSprite.visible = true;

            this.addChild(this._HintSprite);
        }

        createUnshoPWindow() {
            const ww = 480;
            const wh = this.calcWindowHeight(1, true);
            const wy = 6;
            const wx = Graphics.boxWidth - ww - 6;
            const rect = new Rectangle(wx, wy, ww, wh);

            this._unshoPWindow = new window.RSTH_IH.Window_UnshoP(rect);
            this.addWindow(this._unshoPWindow);
            this._unshoPWindow.deactivate();
        }

        createKyoukaUnshoPWindow() {
            const wx = this.margin;
            const wy = 6;
            const ww = Graphics.boxWidth - wx - this._unshoPWindow.width - 12;
            const wh = this.calcWindowHeight(1, true);
            const rect = new Rectangle(wx, wy, ww, wh);

            this._kyoukaUnshoPWindow = new window.RSTH_IH.Window_KyoukaUnshoP(rect);
            this.addWindow(this._kyoukaUnshoPWindow);
            this._kyoukaUnshoPWindow.deactivate();

        }

        createUnshoItemsWindow() {
            const wx = this.margin;
            const wy = this.calcWindowHeight(1, true) + 12;
            const ww = Graphics.boxWidth - wx;
            const wh = Graphics.boxHeight - this.calcWindowHeight(2, true);
            const rect = new Rectangle(wx, wy, ww, wh);
            this._unshoItemsWindow = new window.RSTH_IH.Window_UnshoItems(rect);
            this._unshoItemsWindow.loadGaugeConfig();
            this._unshoItemsWindow.setHandler("ok", () => { });
            this._unshoItemsWindow.setHandler("cancel", () => {
                if (this._unshoItemsWindow._activeIndex !== null) {
                    this._unshoItemsWindow.processCancel();
                } else {
                    this.popScene(); // ← 前のシーンに戻る
                }
            });

            this.addWindow(this._unshoItemsWindow);
        }

        createConfirmWindow() {
            const ww = 300;
            const wh = this.calcWindowHeight(3, true);
            const wx = (Graphics.boxWidth - ww) / 2;
            const wy = (Graphics.boxHeight - wh) / 2;
            const rect = new Rectangle(wx, wy, ww, wh);
            const windowClass = class extends Window_Command {
                makeCommandList() {
                    this.addCommand("はい", "yes");
                    this.addCommand("いいえ", "no");
                }

                windowHeight() {
                    return this.fittingHeight(this.maxItems()) + this.lineHeight();
                }

                itemRect(index) {
                    const rect = super.itemRect(index);
                    rect.y += this.lineHeight(); // ラベル分下にずらす
                    return rect;
                }

                drawAllItems() {
                    this.contents.clear();
                    this.drawText("強化しますか？", 0, 0, this.contentsWidth(), "center");
                    super.drawAllItems(); // 通常項目は下に描画
                }
            };
            this._confirmWindow = new windowClass(rect);
            this._confirmWindow.setHandler("yes", this.onConfirmYes.bind(this));
            this._confirmWindow.setHandler("no", this.onConfirmNo.bind(this));
            this._confirmWindow.openness = 0;
            this.addWindow(this._confirmWindow);
        }

        createSimpleMessageWindow() {
            const ww = 480;
            const wh = this.calcWindowHeight(1, true);
            const wx = (Graphics.boxWidth - ww) / 2;
            const wy = (Graphics.boxHeight - wh) / 2;
            const rect = new Rectangle(wx, wy, ww, wh);

            const windowClass = class extends Window_Selectable {
                initialize(rect) {
                    super.initialize(rect);
                    this._text = "";
                    this._handlers = {};
                    this.hide();
                }

                setText(text) {
                    this._text = text;
                    this.refresh();
                    this.show();
                    this.activate();
                }

                update() {
                    super.update();
                    this.processHandling();
                }

                isOkEnabled() {
                    return true;
                }

                textWidthEx(text) {
                    const tempTextState = { index: 0, text: text, x: 0, y: 0 };
                    return this.drawTextEx(text, 0, this.contents.height + 100); // 仮の描画で幅取得
                }

                processOk() {
                    this.callOkHandler();
                    this.deactivate();
                    this.hide();
                    SceneManager._scene._unshoItemsWindow.activate();
                }

                processCancel() {
                    this.callCancelHandler();
                    this.deactivate();
                    this.hide();
                    SceneManager._scene._unshoItemsWindow.activate();
                }

                refresh() {
                    this.contents.clear();
                    const textWidth = this.textWidthEx(this._text);
                    const x = (this.contentsWidth() - textWidth) / 2;
                    this.drawTextEx(this._text, x, 6);
                }

            };


            this._simpleMessageWindow = new windowClass(rect);
            this._simpleMessageWindow.setHandler("ok", () => { });
            this._simpleMessageWindow.setHandler("cancel", () => { });
            this.addWindow(this._simpleMessageWindow);

        }



        onConfirmYes() {
            const i = this._unshoItemsWindow._lastConfirmedIndex;

            this._unshoItemsWindow.updateKyoukaUnshoP();

            const currentP = window.RSTH_IH.unshoP;
            const costP = window.RSTH_IH.kyoukaUnshoP;

            let symbol = null;
            let value = 0;
            if (this._unshoItemsWindow._cursorCol === 0) {
                symbol = this._unshoItemsWindow._items[i].symbol;
                value = this._unshoItemsWindow._gaugeValues[i];
            } else {
                symbol = this._unshoItemsWindow._skills[i].symbol;
                value = this._unshoItemsWindow._skillGaugeValues[i];
            }

            if (currentP < costP) {
                SoundManager.playBuzzer();
                this._simpleMessageWindow.setText("うんしょポイントが足りません！");
                this._simpleMessageWindow.activate();

                this._confirmWindow.close();
                this._unshoItemsWindow.deactivate();
                return;
            }

            // ポイントを減らす
            window.RSTH_IH.unshoP -= costP;

            ConfigManager.unshoP = window.RSTH_IH.unshoP;
            ConfigManager[`gauge_${symbol}`] = value;
            ConfigManager.save();
            //console.log("window.RSTH_IH.unshoP", window.RSTH_IH.unshoP);
            //console.log("costP", costP);
            //console.log("ConfigManager[`gauge_${symbol}`]", ConfigManager[`gauge_${symbol}`]);

            this._unshoPWindow.refresh();
            // 強化済み段階としてロック

            if (this._unshoItemsWindow._cursorCol === 0) {
                this._unshoItemsWindow._savedGaugeValues[i] = value;
                this._unshoItemsWindow._lockedGaugeValues[i] = value;
            } else {
                this._unshoItemsWindow._savedSkillGaugeValues[i] = value;
                this._unshoItemsWindow._lockedSkillGaugeValues[i] = value;
            }

            this._unshoItemsWindow.updateKyoukaUnshoP();

            this._confirmWindow.close();
            this._unshoItemsWindow._activeIndex = null;
            this._unshoItemsWindow.activate();
        }



        onConfirmNo() {
            const i = this._unshoItemsWindow._lastConfirmedIndex;

            let symbol = null;

            if (this._unshoItemsWindow._cursorCol === 0) {
                symbol = this._unshoItemsWindow._items[i].symbol;
            } else {
                symbol = this._unshoItemsWindow._skills[i].symbol;
            }
            const prev = ConfigManager[`gauge_${symbol}`] ?? 0;
            let rate = 0;

            if (this._unshoItemsWindow._cursorCol === 0) {
                rate = this._unshoItemsWindow._valueRates[symbol] ?? 1;
                this._unshoItemsWindow._gaugeValues[i] = prev;
                this._unshoItemsWindow._items[i].value = prev * rate;
            } else {
                rate = this._unshoItemsWindow._skills[i].value ?? 0;
                this._unshoItemsWindow._skillGaugeValues[i] = prev;
                this._unshoItemsWindow._skills[i].value = rate;
            }

            this._unshoItemsWindow.updateKyoukaUnshoP();
            SoundManager.playCancel();

            this._confirmWindow.close();
            this._unshoItemsWindow._activeIndex = null;
            this._unshoItemsWindow.activate();
        }

    };

    window.RSTH_IH.Window_KyoukaUnshoP = class extends Window_Base {
        constructor(rect) {
            super(rect);
            this.refresh();
        }

        drawIconImage(filename, x, y) {
            const bitmap = this._bitmapCache?.[filename] || ImageManager.loadSystem(filename);
            const targetW = 32;
            const targetH = 32;
            if (bitmap.isReady()) {
                this.contents.blt(bitmap, 0, 0, bitmap.width, bitmap.height, x, y, targetW, targetH);
            } else {
                bitmap.addLoadListener(() => {
                    if (this.contents) {
                        this.contents.blt(bitmap, 0, 0, bitmap.width, bitmap.height, x, y, targetW, targetH);
                        this.refresh(); // 再描画を要求
                    }
                });
            }
        }

        refresh() {
            this.contents.clear();
            window.RSTH_IH.kyoukaUnshoP = window.RSTH_IH.kyoukaUnshoP || 0;
            const text = `強化に必要なうんしょP:`;
            this.drawText(text, this._margin, this._margin, this.innerWidth - this._margin * 2, "left");

            const costIconX = 294;
            const costIconY = (this.innerHeight - 32) / 2;

            this.drawIconImage("reward_icon", costIconX, costIconY);

            const text2 = `x ${window.RSTH_IH.kyoukaUnshoP}`;
            this.drawText(text2, costIconX + 32 + 6, this._margin, 320, "left");
        }
    }

    window.RSTH_IH.Window_UnshoP = class extends Window_Base {
        constructor(rect) {
            super(rect);
            window.RSTH_IH.unshoP = ConfigManager.unshoP;
            this.refresh();
        }

        drawIconImage(filename, x, y) {
            const bitmap = this._bitmapCache?.[filename] || ImageManager.loadSystem(filename);
            const targetW = 32;
            const targetH = 32;
            if (bitmap.isReady()) {
                this.contents.blt(bitmap, 0, 0, bitmap.width, bitmap.height, x, y, targetW, targetH);
            } else {
                bitmap.addLoadListener(() => {
                    if (this.contents) {
                        this.contents.blt(bitmap, 0, 0, bitmap.width, bitmap.height, x, y, targetW, targetH);
                        this.refresh(); // 再描画を要求
                    }
                });
            }
        }

        refresh() {
            this.contents.clear();
            const text = `現在のうんしょP:`;
            this.drawText(text, this._margin, this._margin, this.innerWidth - this._margin * 2, "left");

            const costIconX = 216;
            const costIconY = (this.innerHeight - 32) / 2;

            this.drawIconImage("reward_icon", costIconX, costIconY);

            const text2 = `x ${window.RSTH_IH.unshoP}`;
            this.drawText(text2, costIconX + 32 + 6, this._margin, 320, "left");
        }
    }

    window.RSTH_IH.Window_UnshoItems = class extends Window_Selectable {
        constructor(rect) {
            super(rect);
            this.opacity = 0;
            this.backOpacity = 0;
            this.frameVisible = false;
            this.setBackgroundType(2);

            this._items = [];
            this._skills = [];
            this._gaugeValues = []; // 各項目のゲージ進行度（0〜10）
            this._skillGaugeValues = [];

            this._cursorCol = 0; // 0: 左列, 1: 右列

            this._valueRates = window.RSTH_IH.powerUpValueRates;

            this.setItems();
            this.refresh();
            this.select(0);
            this._activeIndex = null;
            this.createBackgroundSprite();
        }

        initialize(rect) {
            super.initialize(rect);
            this.opacity = 0;
            this.backOpacity = 0;
            this.frameVisible = false;
            this.setBackgroundType(2);

            this._items = [];
            this._skills = [];
            this._gaugeValues = [];
            this._skillGaugeValues = [];

            this._savedGaugeValues = [];
            this._lockedGaugeValues = []; // 現在のセッションで強化確定した段階

            this._savedSkillGaugeValues = [];
            this._lockedSkillGaugeValues = [];

            this._cursorCol = 0;
            this._valueRates = window.RSTH_IH.powerUpValueRates;

            this._gaugeCostTable = {
                normal: [5, 10, 15, 20, 25, 30, 35, 40, 45, 50], // index = 段階 - 1
                lv: [20, 40, 60, 80, 100, 120, 140, 160, 180, 200],
            };


            this._lastConfirmedIndex = null;

            this.setItems();
            this.loadGaugeConfig();

            this.refresh();
            this.select(0);
            this._activeIndex = null;
            this.createBackgroundSprite();
        }

        hitTest(x, y) {
            if (this._activeIndex !== null) return -1;

            const itemsLength = this._items?.length ?? 0;
            const skillsLength = this._skills?.length ?? 0;
            const total = Math.max(itemsLength, skillsLength);

            // 左列のチェック
            if (x < this.contentsWidth() / 2) {
                for (let i = 0; i < itemsLength; i++) {
                    const rect = this.itemRect(i, "left");
                    if (x >= rect.x && x < rect.x + rect.width &&
                        y >= rect.y && y < rect.y + rect.height) {
                        this._cursorCol = 0;
                        this.select(i);
                        return i;
                    }
                }
            }
            // 右列のチェック
            else {
                for (let i = 0; i < skillsLength; i++) {
                    const rect = this.itemRect(i, "right");
                    if (x >= rect.x && x < rect.x + rect.width &&
                        y >= rect.y && y < rect.y + rect.height) {
                        this._cursorCol = 1;
                        this.select(i);
                        return i;
                    }
                }
            }

            return -1;
        }


        loadGaugeConfig() {
            // 左列：通常パラメータ
            for (let i = 0; i < this._items.length; i++) {
                const symbol = this._items[i].symbol;
                const val = ConfigManager[`gauge_${symbol}`];
                if (typeof val === "number") {
                    this._gaugeValues[i] = val;
                    const rate = this._valueRates[symbol] ?? 1;
                    this._items[i].value = val * rate;
                }
            }
            this._savedGaugeValues = [...this._gaugeValues];
            this._lockedGaugeValues = [...this._gaugeValues];

            // 右列：スキルパラメータ
            for (let i = 0; i < this._skills.length; i++) {
                const symbol = this._skills[i].symbol;
                const val = ConfigManager[`gauge_${symbol}`];
                if (typeof val === "number") {
                    this._skillGaugeValues[i] = val;
                }
            }
            this._savedSkillGaugeValues = [...this._skillGaugeValues];
            this._lockedSkillGaugeValues = [...this._skillGaugeValues];
        }



        createBackgroundSprite() {
            const bitmap = new Bitmap(this.width, this.height);
            bitmap.fillAll("#0000009a");
            this._backgroundSprite = new Sprite(bitmap);
            this._backgroundSprite.alpha = 0.4;
            this.addChildToBack(this._backgroundSprite);
        }

        maxItems() {
            // 選択中の列に応じた有効な最大数だけ返す
            if (this._cursorCol === 0) {
                return this._items?.length ?? 0;
            } else {
                return this._skills?.length ?? 0;
            }
        }

        setItems() {
            if (this._items.length > 0) return; // 既に初期化済みなら再代入しない

            const config = {
                HP: { label: "初期MaxHP + ", value: 0 },
                Str: { label: "初期Str + ", value: 0 },
                Vit: { label: "初期Vit + ", value: 0 },
                Dex: { label: "初期Dex + ", value: 0 },
                Agi: { label: "初期Agi + ", value: 0 },
                Luk: { label: "初期Luk + ", value: 0 },
                Drop: { label: "ドロップアイテム吸着距離 + ", value: 0 },
                Lv: { label: "初期Lv + ", value: 0 },
                MaxLv: { label: "Lv上限 + ", value: 0 },
                Zeny: { label: "初期Zeny + ", value: 0 },
            };

            this._items = Object.entries(config).map(([symbol, data]) => ({
                symbol,
                label: data.label,
                value: data.value
            }));
            this._gaugeValues = new Array(this._items.length).fill(0);

            const addskills = {
                KB: { label: "ノックバック耐性", value: 300 },
                HS: { label: "ヒットストップ耐性", value: 300 },
                IG: { label: "熱い！！状態耐性", value: 300 },
                ST1: { label: "毒、猛毒、混乱耐性", value: 300 },
                ST2: { label: "麻痺耐性", value: 200 },
                DFL: { label: "床ダメージ無効", value: 300 },
                RG: { label: "HP自動回復(0.5秒毎に50回復)", value: 400 },
                RV: { label: "自動復活(1プレイにつき1回発動)", value: 400 },
            }
            this._skills = Object.entries(addskills).map(([symbol, data]) => ({
                symbol,
                label: data.label,
                value: data.value
            }));
            this._skillGaugeValues = new Array(this._skills.length).fill(0);

            this.refresh();
        }

        drawItemLeft(index) {
            const data = this._items[index];
            if (!data) return;

            const rect = this.itemRect(index, "left");

            // カーソル位置にハイライト
            if (this.index() === index && this._cursorCol === 0) {
                this.drawRectHighlight(rect);
            }


            const lineH = this.lineHeight();
            const text = `${data.label}${String(data.value).padStart(1, "0")}`;
            this.drawText(text, rect.x + 6, rect.y, rect.width - 6, "left");

            const gaugeX = rect.x + 10;
            const gaugeY = rect.y + lineH - 2;
            const boxW = 16;
            const boxH = 28;
            const margin = 4;
            const value = this._gaugeValues[index];

            for (let i = 0; i < 10; i++) {
                const x = gaugeX + i * (boxW + margin);
                this.contents.fillRect(x, gaugeY, boxW, boxH, "#121922ff");
                if (i < value) {
                    this.contents.fillRect(x + 2, gaugeY + 2, boxW - 4, boxH - 4, "#4193ffff");
                }
            }
        }

        drawItemRight(index) {
            const data = this._skills[index];
            if (!data) return;

            const rect = this.itemRect(index, "right");

            // カーソル位置にハイライト
            if (this.index() === index && this._cursorCol === 1) {
                this.drawRectHighlight(rect);
            }


            const lineH = this.lineHeight();
            this.drawText(data.label, rect.x + 6, rect.y, rect.width - 6, "left");


            const gaugeX = rect.x + 10;
            const gaugeY = rect.y + lineH - 2;
            const gaugeW = 196;
            const gaugeH = 28;
            const gaugeValue = this._skillGaugeValues[index] ?? 0;

            this.contents.fillRect(gaugeX, gaugeY, gaugeW, gaugeH, "#121922ff");
            if (gaugeValue >= 1) {
                this.contents.fillRect(gaugeX + 2, gaugeY + 2, gaugeW - 4, gaugeH - 4, "#4193ffff");
            }

        }


        drawRectHighlight(rect) {
            this.contents.fillRect(rect.x, rect.y, rect.width, rect.height, "#ffffff40");
        }


        itemRect(index, side = "left") {
            const rect = new Rectangle();
            rect.width = this.contentsWidth() / 2;
            rect.height = this.lineHeight() * 2;
            rect.x = (side === "left") ? 0 : this.contentsWidth() / 2;
            rect.y = index * rect.height + 6;
            return rect;
        }


        processOk() {
            const i = this.index();
            if (this._cursorCol === 0) {
                if (this._activeIndex === i) {
                    this._lastConfirmedIndex = i;
                    SceneManager._scene._confirmWindow.open();
                    SceneManager._scene._confirmWindow.activate();
                    this.deactivate();
                } else {
                    this._activeIndex = i;
                    this._initialGaugeValue = this._gaugeValues[i];
                }
            } else {
                if (this._activeIndex === i) {
                    this._lastConfirmedIndex = i;
                    SceneManager._scene._confirmWindow.open();
                    SceneManager._scene._confirmWindow.activate();
                    this.deactivate();
                } else {
                    this._activeIndex = i;
                    this._initialGaugeValue = this._skillGaugeValues[i];
                }
            }
            this.playOkSound();
            this.refresh();
            this.setCursorRectByCol();
        }



        processCancel() {
            if (this._activeIndex !== null) {
                const i = this._activeIndex;

                let symbol = null;
                let prev = 0;
                let rate = 0;
                let locked = 0;

                if (this._cursorCol === 0) {
                    symbol = this._items[i].symbol;
                    prev = ConfigManager[`gauge_${symbol}`] ?? 0;

                    rate = this._valueRates[symbol] ?? 1;
                    locked = this._lockedGaugeValues[i] ?? 0;
                    // ロックされた段階より下には戻せない
                    const valueToSet = Math.max(prev, locked);

                    this._gaugeValues[i] = valueToSet;
                    this._items[i].value = valueToSet * rate;
                } else {
                    symbol = this._skills[i].symbol;
                    prev = ConfigManager[`gauge_${symbol}`] ?? 0;

                    rate = this._skills[i].value ?? 0;
                    locked = this._lockedSkillGaugeValues[i] ?? 0;
                    // ロックされた段階より下には戻せない
                    const valueToSet = Math.max(prev, locked);

                    if (RSTH_DEBUG_LOG) console.warn("valueToSet", valueToSet);
                    this._skillGaugeValues[i] = valueToSet;
                    this._skills[i].value = rate;
                }

                SoundManager.playCancel();
                this._activeIndex = null;
                this.updateKyoukaUnshoP();
                this.refresh();
            } else {
                SoundManager.playCancel();
                this.callCancelHandler();
            }
            this.setCursorRectByCol();
        }

        updateKyoukaUnshoP() {
            let total = 0;

            for (let i = 0; i < this._gaugeValues.length; i++) {
                const symbol = this._items[i].symbol;
                const gauge = this._gaugeValues[i];
                const base = this._savedGaugeValues[i] ?? 0;

                const isLv = (symbol === "Lv" || symbol === "MaxLv");
                const table = isLv ? this._gaugeCostTable.lv : this._gaugeCostTable.normal;

                for (let step = base; step < gauge; step++) {
                    total += table[step] ?? 999; // 安全策として未知段階は999
                }
            }

            for (let i = 0; i < this._skillGaugeValues.length; i++) {
                const gauge = this._skillGaugeValues[i];
                const base = this._savedSkillGaugeValues[i] ?? 0;

                const cost = this._skills[i].value ?? 0; // addskillsから取得したvalue

                for (let step = base; step < gauge; step++) {
                    total += cost;
                }
            }


            window.RSTH_IH.kyoukaUnshoP = total;

            if (SceneManager._scene && SceneManager._scene._kyoukaUnshoPWindow) {
                SceneManager._scene._kyoukaUnshoPWindow.refresh();
            }
        }



        cursorUp(wrap = false) {
            if (this._activeIndex !== null) return; // 固定中は無効

            this.playCursorSound();
            super.cursorUp(wrap);
        }

        cursorDown(wrap = false) {
            if (this._activeIndex !== null) return; // 固定中は無効
            this.playCursorSound();
            super.cursorDown(wrap);
        }

        cursorLeft(wrap = false) {
            if (this._activeIndex !== null) {
                if (this._cursorCol === 0) {
                    const i = this.index();
                    const saved = this._savedGaugeValues[i] ?? 0;
                    const locked = this._lockedGaugeValues[i] ?? 0;
                    this._gaugeValues[i] = Math.max(saved, locked, this._gaugeValues[i] - 1);

                    const symbol = this._items[i].symbol;
                    const rate = this._valueRates[symbol] ?? 1;
                    this._items[i].value = this._gaugeValues[i] * rate;
                    this.refresh();
                    this.updateKyoukaUnshoP();
                    this.playCursorSound();
                } else if (this._cursorCol === 1) {
                    const i = this.index();
                    const saved = this._savedSkillGaugeValues[i] ?? 0;
                    const locked = this._lockedSkillGaugeValues[i] ?? 0;
                    this._skillGaugeValues[i] = Math.max(saved, locked, this._skillGaugeValues[i] - 1);

                    this.refresh();
                    this.updateKyoukaUnshoP();
                    this.playCursorSound();
                }

                return;
            }

            if (this._cursorCol === 1) {
                const max = this._items?.length ?? 0;
                if (this.index() < max) {
                    this._cursorCol = 0;
                    this.refresh();
                    this.updateKyoukaUnshoP();
                    this.playCursorSound();
                    return;
                }
            }

            super.cursorLeft(wrap);
        }

        cursorRight(wrap = false) {
            if (this._activeIndex !== null) {
                if (this._cursorCol === 0) {
                    const i = this.index();

                    this._gaugeValues[i] = Math.min(10, this._gaugeValues[i] + 1);

                    const symbol = this._items[i].symbol;
                    const rate = this._valueRates[symbol] ?? 1;
                    this._items[i].value = this._gaugeValues[i] * rate;
                    this.refresh();
                    this.updateKyoukaUnshoP();
                    this.playCursorSound();
                } else if (this._cursorCol === 1) {
                    const i = this.index();

                    const saved = this._savedSkillGaugeValues[i] ?? 0;
                    const locked = this._lockedSkillGaugeValues[i] ?? 0;

                    this._skillGaugeValues[i] = Math.max(saved, locked, 1); // 1段階固定


                    this.refresh();
                    this.updateKyoukaUnshoP();
                    this.playCursorSound();
                }

                return;
            }

            if (this._cursorCol === 0) {
                const max = this._skills?.length ?? 0;
                if (this.index() < max) {
                    this._cursorCol = 1;
                    this.refresh();
                    this.updateKyoukaUnshoP();
                    this.playCursorSound();
                    return;
                }
            }

            super.cursorRight(wrap);
        }


        drawAllItems() {
            this.contents.clear();
            const itemsLength = Array.isArray(this._items) ? this._items.length : 0;
            const skillsLength = Array.isArray(this._skills) ? this._skills.length : 0;
            const total = Math.max(itemsLength, skillsLength); // ← 両方が未定義でも安全

            for (let i = 0; i < total; i++) {
                if (i < itemsLength) this.drawItemLeft(i);
                if (i < skillsLength) this.drawItemRight(i);
            }
            this.setCursorRectByCol();
        }

        setCursorRectByCol() {
            if (this._activeIndex !== null) {
                // 固定中はカーソル非表示（点滅なし）
                this.setCursorRect(0, 0, 0, 0);
                return;
            }

            const index = this.index();
            const side = this._cursorCol === 0 ? "left" : "right";
            const rect = this.itemRect(index, side);
            this.setCursorRect(rect.x, rect.y, rect.width, rect.height);
        }


        select(index) {
            const max = this.maxItems();
            const clampedIndex = Math.min(index, max - 1); // ← 選択範囲を制限
            super.select(clampedIndex);
            this.setCursorRectByCol();
            this.refresh();
        }



    }

    // 床ダメージ処理を上書き
    Game_Actor.prototype.checkFloorEffect = function () {
        if ($gamePlayer.isOnDamageFloor() &&
            ConfigManager["gauge_DFL"] !== 1) {
            this.executeFloorDamage();
        }
    };


})();